import { useState, useEffect } from "react";
import { useSession } from "next-auth/react";
import { useRouter } from "next/router";
import { useLanguage } from "@/contexts/LanguageContext";
import TabNavigation from "@/components/analysis/TabNavigation";
import OverviewTab from "@/components/analysis/OverviewTab";
import CostsTab from "@/components/analysis/CostsTab";
import PerformanceTab from "@/components/analysis/PerformanceTab";
import DepartmentsTab from "@/components/analysis/DepartmentsTab";
import KPITab from "@/components/analysis/KPITab";
import {
  Technician,
  CostAnalysisData,
  SystemAnalysisData,
  TabId,
} from "@/types/analysis";

export default function CostAnalysis() {
  const { data: session, status } = useSession();
  const { t } = useLanguage();
  const router = useRouter();
  const [technicians, setTechnicians] = useState<Technician[]>([]);
  const [costData, setCostData] = useState<CostAnalysisData | null>(null);
  const [systemData, setSystemData] = useState<SystemAnalysisData | null>(null);
  const [loading, setLoading] = useState(true);
  const [selectedTechnician, setSelectedTechnician] = useState<string>("");
  const [startDate, setStartDate] = useState("");
  const [endDate, setEndDate] = useState("");
  const [activeTab, setActiveTab] = useState<TabId>("overview");

  useEffect(() => {
    if (status === "loading") return;
    if (!session || (session.user as any)?.role !== "ADMIN") {
      router.push("/");
      return;
    }
    fetchData();
  }, [session, status, router]);

  const fetchData = async () => {
    try {
      setLoading(true);
      const [techniciansRes, costRes, systemRes] = await Promise.all([
        fetch("/api/admin/technicians/pricing"),
        fetch("/api/admin/cost-analysis"),
        fetch("/api/admin/system-analysis"),
      ]);

      if (techniciansRes.ok) {
        const techniciansData = await techniciansRes.json();
        setTechnicians(techniciansData.technicians);
      }

      if (costRes.ok) {
        const costAnalysisData = await costRes.json();
        setCostData(costAnalysisData);
      }

      if (systemRes.ok) {
        const systemAnalysisData = await systemRes.json();
        setSystemData(systemAnalysisData);
      }
    } catch (error) {
      console.error("Error fetching data:", error);
    } finally {
      setLoading(false);
    }
  };

  const handleFilter = async () => {
    try {
      setLoading(true);
      const params = new URLSearchParams();
      if (selectedTechnician) params.append("technicianId", selectedTechnician);
      if (startDate) params.append("startDate", startDate);
      if (endDate) params.append("endDate", endDate);

      const response = await fetch(`/api/admin/cost-analysis?${params}`);
      if (response.ok) {
        const data = await response.json();
        setCostData(data);
      }
    } catch (error) {
      console.error("Error filtering data:", error);
    } finally {
      setLoading(false);
    }
  };

  if (status === "loading" || loading) {
    return (
      <div className='min-h-screen bg-gray-50'>
        <div className='container mx-auto px-4 py-8'>
          <div className='text-center'>{t("common.loading")}</div>
        </div>
      </div>
    );
  }

  return (
    <div className='min-h-screen bg-gray-50'>
      <div className='container mx-auto px-4 sm:px-6 py-6 sm:py-8'>
        <div className='mb-6 sm:mb-8'>
          <h1 className='text-2xl sm:text-3xl font-bold text-gray-900 mb-2'>
            Analysis
          </h1>
          <p className='text-sm sm:text-base text-gray-600'>
            Comprehensive analysis of tickets, performance, costs, and system
            metrics
          </p>
        </div>

        <TabNavigation activeTab={activeTab} onTabChange={setActiveTab} />

        {activeTab === "overview" && systemData && (
          <OverviewTab systemData={systemData} />
        )}

        {activeTab === "costs" && costData && (
          <CostsTab
            costData={costData}
            technicians={technicians}
            selectedTechnician={selectedTechnician}
            setSelectedTechnician={setSelectedTechnician}
            startDate={startDate}
            setStartDate={setStartDate}
            endDate={endDate}
            setEndDate={setEndDate}
            onFilter={handleFilter}
          />
        )}

        {activeTab === "performance" && systemData && (
          <PerformanceTab systemData={systemData} />
        )}

        {activeTab === "departments" && systemData && (
          <DepartmentsTab systemData={systemData} />
        )}

        {activeTab === "kpi" && systemData && (
          <KPITab systemData={systemData} />
        )}
      </div>
    </div>
  );
}
